<?php
// public utilities

/** 
 * SEND AN E-MAIL
 * @param to_name 			= Name of the receiver
 * @param to_mail			= E-mail of the receiver (could be multiple receivers split by comma)
 * @param subj 				= Subject of the e-mail
 * @param txt 				= Content of the e-mail
 * @param from_name 		= Name of the sender (false to use the one specified in the settings)
 * @param from_mail 		= E-mail of the sender (false to use the one specified in the settings)
 * @param reply_to_name 	= Name for the "to reply" function (false to use the one specified in the settings)
 * @param reply_to_mail 	= E-mail for the "to reply" function (false to use the one specified in the settings)
 
 * @param (array) attach 	= E-mail attachment 
                                - $_FILES resource (must have the "tmp_name" and "name" indexes)
                                - or manual associative array('file_data' => file contents that could be written in a file, 'file_name' => example.pdf)
                                - or manual associative array having file path as key and filename as value
 *
 * @return (bool)
 */
function pcma_send_mail($to_name, $to_mail, $subj, $txt, $from_name = false, $from_mail = false, $reply_to_name = false, $reply_to_mail = false, $attach = array()) {

	// get basic data to send mail
	if(!$from_name) {$from_name = get_option('pcma_from_name');}
	if(!$from_mail) {$from_mail = get_option('pcma_from_mail');}
	if(!$reply_to_name) {$reply_to_name = get_option('pcma_reply_to_name');}
	if(!$reply_to_mail) {$reply_to_mail = get_option('pcma_reply_to_mail');}
	
	// check basic data
	if(!$from_name || !$from_mail || !$reply_to_name || !$reply_to_mail) {
		trigger_error('One or more basic data are missing');
        return false;
	}
    
    
    // manage multiple receivers split by comma
    if(strpos($to_mail, ',') !== false) {
        $to_mail = str_replace(' ', '', $to_mail);
        $receivers = explode(',', $to_mail);
        
        $response = true;
        foreach($receivers as $rec) {
            $the_name = (empty($to_name)) ? $rec : $to_name;
            $response = pcma_send_mail($the_name, $rec, $subj, $txt, $from_name, $from_mail, $reply_to_name, $reply_to_mail, $attach);
        }
        
        return $response;
    }
    
    
    // final check over e-mail address
    if(!filter_var($to_mail, FILTER_VALIDATE_EMAIL)) {
        trigger_error($to_mail.' is not a valid e-mail');
        return false;    
    }
        
	
	$txt = (strpos(strtolower($txt), '<html') !== false) ? $txt : '<html><body>'. wpautop($txt) .'</body></html>';	
	$txt = do_shortcode($txt);

	$GLOBALS['pcma_send_mail'] = array(
		'to_name'	=> $to_name,
		'to_mail'	=> $to_mail,
		'txt'		=> $txt,
		'from_name'	=> $from_name,
		'from_mail'	=> $from_mail,
		'reply_to_name'	=> $reply_to_name,
		'reply_to_mail'	=> $reply_to_mail,
		'attach'		=> $attach	
	);
	
	
	add_filter('wp_mail_content_type', 'pcma_html_mail_content_type', 9999);
	add_action('phpmailer_init', 'pcma_setup_phpmailer', 9999);
	
	$headers = array('Content-Type: text/html; charset=UTF-8');
    
    
    include_once(PCMA_DIR .'/classes/mail_error_debug.php');
    $debugger = new pcma_wp_mail_errors();
	$response = wp_mail($to_mail, $subj, $txt, $headers);
	
	remove_filter('wp_mail_content_type', 'pcma_html_mail_content_type', 9999);
	remove_action('phpmailer_init', 'pcma_setup_phpmailer', 9999);
	
    if(!$response) {
        $debugger->get_error();    
    }
	return $response;
}



// force mail to be in HTML
function pcma_html_mail_content_type() {
    return 'text/html';
}



// Mail Actions custom WP PHPMailer setup
function pcma_setup_phpmailer($phpmailer){
	$mdata = $GLOBALS['pcma_send_mail'];
	
	// from
	$phpmailer->SetFrom($mdata['from_mail'], $mdata['from_name'], 0);
	
	// to
	if(!is_array($mdata['to_mail']) && strpos($mdata['to_mail'], ',') !== false) {
		$mdata['to_mail'] = explode(',', $mdata['to_mail']);	
	}
	
	if(is_array($mdata['to_mail'])) {
		foreach($mdata['to_mail'] as $single_mail) {
			$phpmailer->AddAddress( trim($single_mail) );
		}
	} else {
		$phpmailer->AddAddress($mdata['to_mail'], $mdata['to_name']);
	}
	
	// reply to
	$phpmailer->AddReplyTo($mdata['reply_to_mail'], $mdata['reply_to_name']);
	
	// mail charset
	$phpmailer->CharSet = "utf-8";
	
	// attachment
	if(!empty($mdata['attach'])) {
        foreach($mdata['attach'] as $key => $val) {
            
            if(isset($val['tmp_name'])) {
                $phpmailer->AddAttachment($val['tmp_name'], $val['name']);
            }
            elseif(isset($val['file_data'])) {
                $phpmailer->addStringAttachment($val['file_data'], $val['file_name']);
            }
            else {
                $phpmailer->AddAttachment($key, $val);        
            }
        }
	}
			
	
	// SMTP
	$use_smtp = pcma_use_smtp();
	if($use_smtp) {
        $phpmailer->IsSMTP();
        
		$phpmailer->Host = $use_smtp['host'];
		$phpmailer->Username = $use_smtp['user'];
		$phpmailer->Password = $use_smtp['psw'];
        
        $phpmailer->Port = (int)get_option('pcma_smtp_port'); 

		if(get_option('pcma_use_smtp_auth')) {
            $phpmailer->SMTPAuth = true;
        }
        if(get_option('pcma_smtp_secure') != 'none') {
            $phpmailer->SMTPSecure = get_option('pcma_smtp_secure');
        }
        if(get_option('pcma_smtp_autotls')) {
            $phpmailer->SMTPAutoTLS = true;
        }
	}
    
    // $phpmailer->SMTPDebug = 1;
}

